/*
 * ParameterFile.cpp
 *
 */
#include "ParameterFile.h"

//default constructor
ParameterFile::ParameterFile() {
  file_name = DEF_FILE_NAME;
  output_dir = DEF_FILE_NAME;
  
  // set uniform background
  background_type = UNIFORM;
  background.resize(ALPH_NUM, DEF_NC_FREQ);
  background_is_set = false;
  
  lowest_score = DEF_LOW_SCORE;
  precision = DEF_PRECISION;
  paddingfrac = DEF_PADDING;
  lambda = DEF_LAMBDA; // the default value is set to 1/8 to make have WM score
      
  mode = DWT;
  train_DWT = false;
}

ParameterFile::ParameterFile (string file){
  // check that we have a parameter file to read
  file_name = file;
  
  // if we don't get a parameter file, we can quit right here
  ifstream  param_file (file_name.c_str(), ios::in);
  if (!param_file.is_open()){
    cerr << "The parameter file could not be opened." << endl;
    exit(1);
  }

  /* Set all values to default to start with */
  // CHECK 
  background_is_set = false;	// if background is not mentioned in param_file

  // Initialize the background to uniform and reset it if the parameter file
  // specifies that the background should be set from input sequences
  background_type = UNIFORM;
  background.resize(ALPH_NUM, DEF_NC_FREQ); 

  // CHECK (in the .h file min was -50 here it was -80
  lowest_score = DEF_LOW_SCORE;	// if lowest_score is not mentioned in param_file
  precision = DEF_PRECISION;
  paddingfrac = DEF_PADDING; // Default no padding 
  lambda = DEF_LAMBDA; // the default value is set to 1/8 to make have WM score
 
  output_dir = DEF_FILE_NAME; // will have to be read

  mode = DWT; // will have to be read
  
  train_DWT = false; // will have to be specifically set


  do{
    string temp;
    param_file >> temp;
    if (temp.find('#') < temp.length()){
      // if we want to set the background from input sequences
      // we would need to specify this in the background_type parameter
      if (temp.find("background_type") < temp.length()) {
	initialize_background_type(param_file);
	background_is_set = true;
      }
      // we also leave the possibility to give an arbitrary background
      if (temp.find("background_probs") < temp.length()) {
	initialize_background(param_file);
	background_is_set = true;
      }
      // output_directory
      if (temp.find("output_dir") < temp.length()) {
	initialize_output_dir(param_file);
      }
      if (temp.find("mode") < temp.length()) {
	initialize_mode(param_file);
      }
      if (temp.find("train_DWT") < temp.length()) {
	initialize_train_DWT(param_file);
      }
      if(temp.find("precision") < temp.length()){
	initialize_precision(param_file);
      }
      if(temp.find("padding_fraction") < temp.length()){
	initialize_paddingfrac(param_file);
      }
      if(temp.find("minimum_score") < temp.length()){
	  initialize_minimum_score(param_file);
	}
    }
    if (temp.find("//") < temp.length()){
      while(temp.find('\n') < temp.length()) {
	param_file >> temp;
      }
    }
  }
  while (! param_file.eof());
  
  // sanity check on what has been read from the parameter file
  if(output_dir == DEF_FILE_NAME) {
    cerr << "No output directory was specified. Writing to current directory..." << endl;
    output_dir = "";
  }
  }


ParameterFile::~ParameterFile() {
	// TODO Auto-generated destructor stub
}

void ParameterFile::initialize_background_type(ifstream & param_file){
  // by default background is uniform
  // this variable tells us whether we set it from the parameter file
  // or from the input sequences
  string temp;
  param_file >> temp;
  switch (atoi(temp.c_str())) {
      case UNIFORM:
	// nothing to do, we already have a uniform background by default
	break;
      case FROMPARAM:
	// we expect to set the background from this parameter file
	// but if we don't find the appropriate directives it will just be uniform
	background_type = FROMPARAM;
	break;
      case FROMSEQ:
	// we want to set the background from the input sequences
	background_type = FROMSEQ;
	// CHECK if we need to reinforce than that the default settings are uniform
	break;
      default:
	cerr << "The background type can only be one of 0 (uniform), 1 (set from the parameter file) or 2 (set from the input sequences)" << endl;
	exit(1);
    }
}
  
// CHECK if we ever need this
void ParameterFile::initialize_background(ifstream & param_file){
  // space and defauly values are already set in the constructor

  string temp;
  int index = 0;
  double probability = .0;

  for (unsigned short int i = 0; i < ALPH_NUM; i++){
    param_file >> temp;
    index = convert(temp.c_str()[0]);
    if (index == -1){
      cerr << "Character not recognized: " << temp.c_str()[0] << endl;
      exit(1);
    }
    param_file >> temp;
    probability = atof(temp.c_str());
    background[index] = probability;
  }

  // make sure that the background is normalized
  float sum = 0.;
  for (unsigned short int i = 0 ; i < ALPH_NUM ; i++)
    sum += background[i];
  for (unsigned short int i = 0 ; i < ALPH_NUM ; i++)
    background[i] /= sum;

}

void ParameterFile::initialize_output_dir(ifstream & param_file){
  string temp;
  param_file >> temp;
  if (temp.length() > 0)
    output_dir = temp.c_str();
  else
    cerr << "The string provided for the resulting file name is not valid" << endl;
}

void ParameterFile::initialize_precision(ifstream & param_file){
  string temp;
  param_file >> temp;
  precision = atof(temp.c_str());
  if (precision <= 0) {
    cerr << "The precision value in parameter file is not valid and will be set to default" << endl;
    // it's set already
  }
}

void ParameterFile::initialize_paddingfrac(ifstream & param_file){
   string temp;
   param_file >> temp;
   paddingfrac = atof(temp.c_str());
   if (paddingfrac >= 0.9) {
     cerr << "The maximal padding fraction is 90%  value in parameter file is not valid. We will set 90% of motif length" << endl;
     paddingfrac = 0.9;
   }
   if(paddingfrac < 0){
     cerr << "The padding fraction must be non-negative. The value in the parameter file was not valid and we reset it to the default of zero." << endl;
   }
}

void ParameterFile::initialize_minimum_score(ifstream & param_file){
	string temp;
	param_file >> temp;
	// CHECK: no check needs to done to make sure that this value makes sense?
	lowest_score = atof(temp.c_str());
}

void ParameterFile::initialize_mode(ifstream & param_file){
  // set the type of run that we want to do
  // can only be one of 1 (DWT), 2 (ADJ) or 3 (PWM)
  string temp;
  param_file >> temp;

  if(temp == "DWT") {
    mode = DWT;
  }
  else if(temp == "ADJ") {
    mode = ADJ;
  }
  else if(temp == "PWM") {
    mode = PWM;
  }
  else {
    cerr << "Mode can only be one of DWT or ADJ or PWM. Quitting..." << endl;
    exit(1);
  }
}

void ParameterFile::initialize_train_DWT(ifstream & param_file){
  string temp;
  param_file >> temp;
  switch (atoi(temp.c_str())) {
    case 1:
      train_DWT = true;
      break;
    case 0:
      train_DWT = false;
      break;
    default:
      cerr << "train_DWT can only be 0 (false) or 1 (true). I got:" << atoi(temp.c_str())  << endl;
      exit(1);
  }
}

BgType ParameterFile::give_background_type() {
  return background_type;
}

Mode ParameterFile::give_mode() {
  return mode;
}

bool ParameterFile::give_train_DWT() {
  return train_DWT;
}

double ParameterFile::give_paddingfrac() {
  return paddingfrac;
}

double ParameterFile::give_lambda() {
  return lambda;
}

double ParameterFile::give_precision(){
  return precision;
}

string ParameterFile::give_output_dir(){
  return output_dir;
}

std::vector <float> ParameterFile::give_background(){
  return background;
}

double ParameterFile::give_lowest_score(){
  return lowest_score;
}

void ParameterFile::change_background(std::vector <float> probs){
  if (probs.size() == ALPH_NUM){
    for(int i = 0; i < ALPH_NUM; i++) {
      background[i] = probs[i];
    }
  } else{
    cerr << "The vector which was provided for changing background probabilities is inappropriate." << endl;
  }

  // make sure that the background is normalized
  float sum = 0.;
  for (unsigned short int i = 0 ; i < ALPH_NUM ; i++)
    sum += background[i];
  for (unsigned short int i = 0 ; i < ALPH_NUM ; i++)
    background[i] /= sum;
}
